from fastapi import FastAPI, UploadFile, File, Form, HTTPException, BackgroundTasks
from fastapi.responses import FileResponse
from fastapi.staticfiles import StaticFiles
import tempfile
import os
import shutil
import logging

from compress import compress_pdf

app = FastAPI(title="PDF Compressor (single-folder)")
app.mount("/static", StaticFiles(directory="public"), name="static")

@app.get("/")
async def read_index():
    return FileResponse("public/index.html")

@app.post("/compress")
async def compress_endpoint(background_tasks: BackgroundTasks, file: UploadFile = File(...), quality: str = Form("ebook")):
    if not file.filename.lower().endswith('.pdf'):
        raise HTTPException(status_code=400, detail='Only PDF files are allowed')

    with tempfile.TemporaryDirectory() as tmp:
        input_path = os.path.join(tmp, 'input.pdf')
        output_path = os.path.join(tmp, 'output.pdf')
        with open(input_path, 'wb') as f:
            f.write(await file.read())

        ok, msg = compress_pdf(input_path, output_path, quality)
        if not ok:
            logging.error('Compression failed: %s', msg)
            raise HTTPException(status_code=500, detail=f'Compression failed: {msg}')

        final_fd, final_path = tempfile.mkstemp(suffix='.pdf', prefix='compressed_')
        os.close(final_fd)
        shutil.copy2(output_path, final_path)

    background_tasks.add_task(os.unlink, final_path)
    return FileResponse(final_path, media_type='application/pdf', filename=f'compressed_{file.filename}')
