import shutil
import subprocess
import pikepdf
import os
import logging


def _find_gs_exec():
    # Common Windows paths
    windows_paths = [
        r"C:\\Program Files\\gs\\gs10.06.0\\bin\\gswin64c.exe",
        r"C:\\Program Files (x86)\\gs\\gs10.06.0\\bin\\gswin32c.exe",
    ]
    for p in windows_paths:
        if os.path.isfile(p):
            return p

    # Common local paths
    local_candidates = [
        "ghostpdl-10.06.0/bin/gs",
        "ghostpdl-10.06.0/bin/gs.exe",
        "gs/bin/gs",
    ]
    for candidate in local_candidates:
        if os.path.isfile(candidate):
            return candidate

    # PATH
    for exe in ("gs", "gswin64c", "gswin32c"):
        if shutil.which(exe):
            return exe
    return None


def compress_pdf(input_path: str, output_path: str, quality: str = "ebook"):
    settings_map = {"screen": "/screen", "ebook": "/ebook", "printer": "/printer", "prepress": "/prepress"}
    pdf_setting = settings_map.get(quality, "/ebook")

    gs = _find_gs_exec()
    if gs:
        cmd = [
            gs,
            "-sDEVICE=pdfwrite",
            "-dCompatibilityLevel=1.4",
            f"-dPDFSETTINGS={pdf_setting}",
            "-dNOPAUSE",
            "-dQUIET",
            "-dBATCH",
            f"-sOutputFile={output_path}",
            input_path,
        ]
        try:
            subprocess.run(cmd, check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            return True, "Compressed with Ghostscript"
        except subprocess.CalledProcessError as e:
            try:
                stderr = (e.stderr or b"").decode("utf-8", errors="ignore")
            except Exception:
                stderr = str(e)
            logging.warning("Ghostscript failed; falling back to pikepdf. Details: %s", stderr)

    try:
        pdf = pikepdf.Pdf.open(input_path)
        try:
            pdf.save(output_path, compress_streams=True)
        except TypeError:
            pdf.save(output_path)
        return True, "Compressed with pikepdf (basic)"
    except Exception as e:
        logging.exception("pikepdf fallback failed")
        return False, str(e)
